#!perl
# Copyright (c) 2021 SIOS Technology Corp.
########################################################################
#
# Title:       restore for Load Balancer probe reply
#
#
# Usage:    restore -t tagname -i id
#
#           where:
#           "tagname" is the resource tag name
#           "id" is the resource ID
#
# Logging:  All data sent to STDERR will be logged in the lifekeeper log file
#           at the INFO level.

use strict;
use Getopt::Std; # use the standard getopt package
BEGIN { require $ENV{'LKROOT'}.'/etc/default/LifeKeeper.pl'; } # use the LifeKeeper libraries
use IO::Socket::INET;
use IO::Select;
use vars qw($opt_t $opt_i $opt_C);

my $debug = 0;

my $thisCMD=`basename $0`;
chomp $thisCMD;
my $perlCMD = "$ENV{'LKROOT'}\\perl\\bin\\perl.exe";
my $parent = 1;
my $scriptname = "restore";
my $userinfo;
my $tag;
my $id;

my $accept_error_count = 0;
my $accept_timeout_count = 0;
my $last_state = "";

my $HC_TIMEOUT = ($ENV{'HC_TIMEOUT'} =~ /^\d+$/) ? $ENV{'HC_TIMEOUT'} : 60;
my $HC_READ_TIMEOUT = ($ENV{'HC_READ_TIMEOUT'} =~ /^\d+$/) ? $ENV{'HC_READ_TIMEOUT'} : 10;
my $HC_MAX_ERROR = ($ENV{'HC_MAX_ERROR'} =~ /^\d+$/) ? $ENV{'HC_MAX_ERROR'} : 3;

sub parseOptions { # noparams

	getopts('t:i:C');
	$tag = "$opt_t";
	$id = "$opt_i";
	if ($opt_C) {
		$parent = 0;
		$scriptname = "daemon";
	}
};

sub getInfo { # params: tag 

	my $tag = shift;
	@_ = `$ENV{'LKROOT'}/bin/ins_list -t "$tag"`;
	if ( @_ == 0 ){
		printf STDERR "$thisCMD: Internal error ins_list failed\n";
		exit 1;
	}
	@_ = split /\001/, $_[0];
	return ($_[5]);
};

sub checkResourceState { # params: tag 
	my $tag = shift;

	if ($parent) {
		return;
	}

	@_ = `$ENV{'LKROOT'}/bin/ins_list -t "$tag"`;
	if ( @_ == 0 ){
		LogMsg('FRS_MES', 33613, $thisCMD, "Resource deleted");
		exit 1;
	}
	@_ = split /\001/, $_[0];
	my $state = $_[6]; # state is the 7th field
	if ("$last_state" eq "ISP" && "$state" ne "ISP") {
		LogMsg('FRS_WARN', 33614, $thisCMD, "Resource out of service");
                exit 1;
	}
	LogMsg('FRS_DEBUG', 33699, $thisCMD, "Resource state = $state");
	$last_state = $state;
};

sub is_daemon { # params: pid
	use Win32::OLE qw(in);
	my $pid = shift;
	my $wmi = Win32::OLE->GetObject("winmgmts://./root/cimv2");
	my $procs = $wmi->ExecQuery("Select * From Win32_Process Where ProcessId = $pid");

	foreach my $p (in $procs) {
		my $CommandLine = $p->CommandLine;
		my ($exepath, $perlpath) = split(/\s+/, $p->CommandLine);
		my $perlfname = `basename $perlpath`;
		chomp $perlfname;
		if ($exepath eq $perlCMD && $perlfname eq $thisCMD) {
			LogMsg('FRS_DEBUG', 33630, $thisCMD, "The process (PID = $pid) is not a gen-lb daemon. $exepath:$perlCMD $perlfname:$thisCMD CommandLine = $CommandLine");
			return 1;
		}
		else {
			LogMsg('FRS_DEBUG', 33631, $thisCMD, "The process (PID = $pid) is not a gen-lb daemon. CommandLine = $CommandLine");
			return 0;
		}
	}
	LogMsg('FRS_DEBUG', 33632, $thisCMD, "The process (PID = $pid) does not exist.");
	return 0;
}




sub check_exist_process { # params: id
	my $id = shift;
	my $pidfile = "$ENV{'LKROOT'}/config/$id.pid";
	if(open(my $PID, "< $pidfile")) {
		my $pid=<$PID>;
		close($PID);

		if (is_daemon($pid)) {
			LogMsg('FRS_ERR', 33601, $thisCMD, "Daemon already started.");
			exit 1;
		}
		else {
			return;
		}
	}
	else {
		LogMsg('FRS_DEBUG', 33615, $thisCMD, "Daemon not started. (pidfile not found)");
	}
}

sub save_pid { # params: id
	my $id = shift;
	my $pidfile = "$ENV{'LKROOT'}/config/$id.pid";

	if ($parent) {
		return;
	}

	if(open(my $PID, "> $pidfile")) {
		print $PID "$$";
		close($PID);
	}
}



sub handle_accept_timeout { # noparams
	$accept_timeout_count ++;
	if ($accept_timeout_count > 1 && !$parent) {
		return;
	}

	LogMsg('FRS_ERR', 33603, $thisCMD, "Health probe not received.");
	if ($parent) {
		exit 1;
	}
	else {
		my $cmdline="$ENV{'LKROOT'}/bin/flg_create -f \"$id-hc-ng\"";
		system($cmdline);
	}
}

sub handle_accept_error { # params: reason
	my $reason = shift;
	$accept_error_count ++;

	LogMsg('FRS_ERR', 33611, $thisCMD, "Failed to accept the health probe:$reason");

	if ($accept_error_count > $HC_MAX_ERROR) {
		exit 1;
	}
}

sub LogMsg {
	my $category = shift;
	my $msg_code = shift;
	my $procName = shift;
	my $message = shift;

	if ($category eq 'FRS_DEBUG' && $debug == 0) {
		return ;
	}

	`$ENV{'LKROOT'}/bin/lk_err -c $category -n $msg_code -p $procName "gen-lb:$scriptname:$tag $message"`
}


#############################
# Main execution begins here
#############################

# parse command line options
parseOptions;

# $tag and $id are set now if they were given on the command line
($userinfo)=getInfo($tag);

check_exist_process($id);
save_pid($id);

(my $port, my $msg) = split(/ /, $userinfo);

if ($port !~ /^\d+$/ or $port < 1024 or $port > 65535) {
	LogMsg('FRS_ERR', 33612, $thisCMD, "Port number is not valid.");
	exit 1;
}

my $socket = new IO::Socket::INET( Proto => "tcp",
				   LocalPort => $port,
				   Listen => 1,
				   Reuse => 1);
if (!defined $socket) {
	LogMsg('FRS_ERR', 33602, $thisCMD, "Socket open failed.");
	exit 1;
}

# Wait for probe connection.
my $accept_sel = new IO::Select();
my $read_sel = new IO::Select();
$accept_sel->add($socket);

while (1) {
	checkResourceState($tag);
	my @handles = $accept_sel->can_read($HC_TIMEOUT);
	my $handles_num = @handles;
	if ($handles_num == 0) {
		# timeout
		handle_accept_timeout();
		next;
        }
	my $client = $socket->accept();
	if (!defined $client) {
		handle_accept_error($!);
		next;
	}


	if (defined $msg) {
		$read_sel->add($client);
		my @handles = $read_sel->can_read($HC_READ_TIMEOUT);
		my $handles_num = @handles;
		if ($handles_num != 0) {
			# data arrived
			my $request = <$client>;
		}
		print $client $msg;
		$read_sel->remove($client);
	}
	close $client;

	# Probe reply daemon
	$accept_error_count = 0;
	$accept_timeout_count = 0;
	my $cmdline="$ENV{'LKROOT'}/bin/flg_remove -f \"$id-hc-ng\" > NUL 2>&1";
	system($cmdline);
	LogMsg('FRS_DEBUG', 33604, $thisCMD, "Health check received.");

	if ($parent) {
		close($socket);
		my $cmdline="start $perlCMD $0 -t \"$tag\" -i \"$id\" -C";
		system($cmdline);
		exit 0;
	}
}

# never comes here
exit 0;
