#!perl
# Copyright (c) 2021 SIOS Technology Corp.
########################################################################
#
# Title:    Load balancer helth proble
#
# Description:  This script checks the health of a particular resource
#               instance. It should return 0 if the resource is OK or
#               if the resource does not exist or the script fails 
#               for some reason. The script should return <> 0 only in
#               the case of resource failure. The script should invoke
#               sendevent to start local recovery or switchover of a
#               failed resource.
#
# Usage:    quickCheck -t tagname -i id
#
#           where:
#           "tagname" is the resource tag name
#           "id" is the resource ID
#
#           NOTE: quickCheck should not be called directly from restore,
#           but restore should do checks equivalent to a quickCheck
#
# Logging:  All data sent to STDERR will be logged in the lifekeeper
#           log at the INFO level.
#
# ******* IMPORTANT NOTE:
#           QUICKCHECK SCRIPTS ARE RUN PERIODICALLY AND OFTEN.  OUTPUT
#           FROM THIS SCRIPT WILL, THEREFORE, BE LOGGED PERIODICALLY AND
#           OFTEN IN THE LIFEKEEPER LOG.  HANDLE OUTPUT ACCORDINGLY.
#              

use strict;
use Getopt::Std; # use the standard getopt package
BEGIN { require $ENV{'LKROOT'}.'/etc/default/LifeKeeper.pl'; } # use the LifeKeeper libraries
use LK;
use vars qw($opt_t $opt_i);
use Win32::OLE qw(in);

my $debug = 0;

my $thisCMD=`basename $0`;
chomp $thisCMD;
my $perlCMD = "$ENV{'LKROOT'}\\perl\\bin\\perl.exe";
my $scriptname = "quickchk";
my $userinfo;
my $tag;
my $id;

sub parseOptions { # noparams

	getopts('t:i:');
	$tag = "$opt_t";
	$id = "$opt_i";
};

sub checkResourceState { # params: tag 

	my $tag = shift;
	@_ = `$ENV{'LKROOT'}/bin/ins_list -t "$tag"`;
	@_ = split /\001/, $_[0];
	return $_[6]; # state is the 7th field
};

sub is_daemon { # params: pid
	my $pid = shift;
	my $wmi = Win32::OLE->GetObject("winmgmts://./root/cimv2");
	my $procs = $wmi->ExecQuery("Select * From Win32_Process Where ProcessId = $pid");

	foreach my $p (in $procs) {
		my $CommandLine = $p->CommandLine;
		my ($exepath, $perlpath) = split(/\s+/, $p->CommandLine);
		my $perlfname = `basename $perlpath`;
		chomp $perlfname;
		if ($exepath eq $perlCMD && $perlfname eq $thisCMD) {
			LogMsg('FRS_DEBUG', 33636, $thisCMD, "The process (PID = $pid) is a gen-lb daemon. CommandLine = $CommandLine");
			return 1;
		}
		else {
			LogMsg('FRS_DEBUG', 33637, $thisCMD, "The process (PID = $pid) is not a gen-lb daemon. $exepath:$perlCMD $perlfname:$thisCMD CommandLine = $CommandLine");
			return 0;
		}
	}
	LogMsg('FRS_DEBUG', 33638, $thisCMD, "The process (PID = $pid) does not exist.");
	return 0;
}

sub LogMsg {
	my $category = shift;
	my $msg_code = shift;
	my $procName = shift;
	my $message = shift;

	if ($category eq 'FRS_DEBUG' && $debug == 0) {
		return ;
	}

	`$ENV{'LKROOT'}/bin/lk_err -c $category -n $msg_code -p $procName "gen-lb:$scriptname:$tag $message"`
}

#############################
# Main execution begins here
#############################

# parse command line options
parseOptions;

# $tag and $id are set now if they were given on the command line

# now check the resource
my $state = checkResourceState($tag);
if ("$state" ne "ISP") {
	LogMsg('FRS_ERR', 33609, $thisCMD, "Resource out of service");
	exit 0;
}

# Check if daemon is running.
my $pidfile = "$ENV{'LKROOT'}/config/$id.pid";
my $pid;
if(open(my $PID, "< $pidfile")) {
	$pid=<$PID>;
	close($PID);
	if (!is_daemon($pid)) {
		LogMsg('FRS_ERR', 33610, $thisCMD, "Daemon not running.");
		exit 1;
	}
}
else {
	# PID file not found
	LogMsg('FRS_ERR', 33618, $thisCMD, "Failed to read pidfile ($pidfile).");
	exit 1;
}

# Check receive of health check.
if (LK::flg_test('', "$id-hc-ng")) {
	LogMsg('FRS_DEBUG', 33619, $thisCMD, "quickchk completed");
	exit 0;
} else {
	LogMsg('FRS_DEBUG', 33620, $thisCMD, "flg $id-hc-hg is on");
	exit 1;
}
