#!perl
# Copyright (c) 2021 SIOS Technology Corp.
########################################################################
#
# Title:       remove for Load Balancer probe reply
#
#
# Description: This script is called by the LifeKeeper core when a resource is
#              being taken out of service.  This could have been caused by a
#              manual "out of service" operation.
#
# Usage:    remove -t tagname -i id
#
#           where:
#           "tagname" is the resource tag name
#           "id" is the resource ID
#

use strict;
use Getopt::Std; # use the standard getopt package
BEGIN { require $ENV{'LKROOT'}.'/etc/default/LifeKeeper.pl'; } # use the LifeKeeper libraries
use vars qw($opt_t $opt_i);
use Win32::OLE qw(in);

my $debug = 0;

my $thisCMD=`basename $0`;
chomp $thisCMD;
my $perlCMD = "$ENV{'LKROOT'}\\perl\\bin\\perl.exe";
my $scriptname = "remove";
my $userinfo;
my $tag;
my $id;

sub parseOptions { # noparams

	getopts('t:i:');
	$tag = "$opt_t";
	$id = "$opt_i";
};

sub getInfo { # params: tag 

	my $tag = shift;
	@_ = `$ENV{'LKROOT'}/bin/ins_list -t "$tag"`;
	if ( @_ == 0 ){
		printf STDERR "$thisCMD: Internal error ins_list failed\n";
		exit 1;
	}
	@_ = split /\001/, $_[0];
	return ($_[5]);
};

sub is_daemon { # params: pid
	my $pid = shift;
	my $wmi = Win32::OLE->GetObject("winmgmts://./root/cimv2");
	my $procs = $wmi->ExecQuery("Select * From Win32_Process Where ProcessId = $pid");

	foreach my $p (in $procs) {
		my $CommandLine = $p->CommandLine;
		my ($exepath, $perlpath) = split(/\s+/, $p->CommandLine);
		my $perlfname = `basename $perlpath`;
		chomp $perlfname;
		if ($exepath eq $perlCMD && $perlfname eq $thisCMD) {
			LogMsg('FRS_DEBUG', 33633, $thisCMD, "The process (PID = $pid) is a gen-lb daemon. CommandLine = $CommandLine");
			return 1;
		}
		else {
			LogMsg('FRS_DEBUG', 33634, $thisCMD, "The process (PID = $pid) is not a gen-lb daemon. $exepath:$perlCMD $perlfname:$thisCMD CommandLine = $CommandLine");
			return 0;
		}
	}
	LogMsg('FRS_DEBUG', 33635, $thisCMD, "The process (PID = $pid) does not exist.");
	return 0;
}

sub LogMsg {
	my $category = shift;
	my $msg_code = shift;
	my $procName = shift;
	my $message = shift;

	if ($category eq 'FRS_DEBUG' && $debug == 0) {
		return ;
	}

	`$ENV{'LKROOT'}/bin/lk_err -c $category -n $msg_code -p $procName "gen-lb:$scriptname:$tag $message"`
}

#############################
# Main execution begins here
#############################

# parse command line options
parseOptions;

# $tag and $id are set now if they were given on the command line
($userinfo)=getInfo($tag);

# Check if daemon is running.
my $pidfile = "$ENV{'LKROOT'}/config/$id.pid";
my $pid;
if(open(my $PID, "< $pidfile")) {
	$pid=<$PID>;
	close($PID);
	
	if (is_daemon($pid)) {
		system("kill $pid > NUL");
		LogMsg('FRS_DEBUG', 33616, $thisCMD, "Daemon killed.");
	}            
	else {
		LogMsg('FRS_MES', 33608, $thisCMD, "Daemon not started.");
	}
}
else {
	# PID file not found
	LogMsg('FRS_MES', 33617, $thisCMD, "Failed to read pidfile ($pidfile).");
}

unlink $pidfile;
exit 0;
